
\   EXPCOLOR.ASY - Exponential Decay Curve Fitting Example
\
\   Application example of linking the 7912AD or 7912HB with ASYST
\
\   This version works with color graphics in EGA mode.  The analysis
\   is very sensitive to signal shape and may not work with some
\   signals.  Examine the word FIND.DECAY.START for clues to the 
\   sensitivities.
\
\   Copyright (c) 1987, Tektronix, Inc.
\   The program material contained herin is supplied without warranty or
\   representation of any kind.  Tektronix, Inc., assumes no responsibility
\   and shall have no liability, consequential, or otherwise, of any kind
\   arising from the use of this program material or any part thereof.
\
\   Program is designed to support,
\     Tek 7912HB or 7912AD Transient Digitizer
\     Tek 7A29P or 7A16P Programmable Amplifier
\     Tek 7B90P Programmable Timebase
\
\ Set up GPIB communications with 7912AD or 7912HB and
\ programmable plugins (Assumes Primary Address of 3)
\
3 GPIB.DEVICE DIG             \ Digitizer (Tek 7912AD or 7912HB)
0 SECONDARY.ADDRESS
EOI.ON                        \ Terminate on EOI asserted
EOS.OFF                       \ No extra End-of-String character
INTEGER DIM[ 1030 ] ARRAY BIN.DATA  \ Waveform data input buffer
BIN.DATA []GPIB.BUFFER        \ Declare it
BIN.DATA DMA.GPIB.BUFFER      \ Make it available for DMA

3 GPIB.DEVICE AMP             \ Amplifier (Tek 7A16P or 7A29P)
1 SECONDARY.ADDRESS
3 GPIB.DEVICE TB              \ Timebase (Tek 7B90P)
2 SECONDARY.ADDRESS

SEND.INTERFACE.CLEAR          \ Become Controller-In-Charge
REMOTE.ENABLE.ON


\ Digitize and transfer Average to Center waveform and exit
\ with integer waveform data on stack.
\
: ACQ.ATC
   " DIG DAT;ATC;READ ATC" DIG GPIB.WRITE  \ Digitize the data
   ME LISTENER                \ Set up transfer
   DIG TALKER
   DMA.LISTEN                 \ Use DMA for speed
   UNLISTEN                   \ Release GPIB
   UNTALK

   BIN.DATA UNPACK            \ Unpack buffer to integers
   DUP
   SUB[ 4 , 512 , 2 ]         \ Select high bytes
   256 *                      \ Shift them
   SWAP
   SUB[ 5 , 512 , 2 ]         \ Select low bytes
   +                          \ Add them to shifted high bytes
;


\ Auto intensity words.  Strategy is to set a low trace intensity,
\ digitize and read number of vertical values captured.  Intensity is
\ iteratively increased while counting verticals until there are enough
\ to be considered a good trace.

INTEGER SCALAR NVER           \  Number of verticals found

\ This word digitizes data and reads the byte count of the verticals array.
\
: GET.NUM.VER
   " DIG DAT;READ VER" DIG GPIB.WRITE  \ Read verticals array
   ME LISTENER                \ Set up transfer
   DIG TALKER
   STACK.LISTEN DROP          \ Read & discard % header character
   STACK.LISTEN 256 *         \ Shift high byte of byte count
   STACK.LISTEN +             \ Add low byte of byte count
   1 - 2 / NVER :=            \ Store in NVER
   DMA.LISTEN                 \ Quickly read & discard verticals
   UNLISTEN                   \ Release the GPIB
   UNTALK
;

INTEGER SCALAR MAI            \ Main intensity
INTEGER SCALAR INTENS.START   \ Starting value for intensity
INTEGER SCALAR INTENS.INCR    \ Increment for intensity increase
INTEGER SCALAR NVER.MIN       \ Min number of vert values accepted

 250 INTENS.START :=          \ Initialize variables
   2 INTENS.INCR :=
1000 NVER.MIN :=

\ This word iteratively determines a good trace intensity and
\ leaves the digitizer set at that level.
\
: AUTO.INTENSITY
   " MAI 0" DIG GPIB.WRITE    \ Turn off trace
   75 MSEC.DELAY              \ Let target erase
   INTENS.START MAI :=        \ Get a starting intensity
   BEGIN
     MAI INTENS.INCR + DUP MAI :=      \ Increment the intensity
     " MAI" "." "CAT DIG GPIB.WRITE    \ Set it in digitizer
     GET.NUM.VER              \ Digitize & find out how many verticals
     NVER NVER.MIN >          \ Enough?
   UNTIL                      \ Loop until enough trace intensity
;

REAL SCALAR GND.LEVEL         \ Mean ground level
100 STRING "AMP.SETS          \ Buffer for amplifier settings
100 STRING "TB.SETS           \ Buffer for timebase settings

\ Automatic ground reference word.  Assumes programmable plugins.
\ Queries both plugins for settings, grounds amplifier and auto-triggers
\ timebase on line signal.  AUTO.INTENSITY sets trace brightness and
\ ACQ.ATC acquires the ground signal, which is reduced to an average value.
\
: GET.GND.LEVEL
   " SET?" AMP GPIB.WRITE     \ Store amplifier settings
   "AMP.SETS AMP GPIB.READ   
   " SET?" TB GPIB.WRITE      \ Store timebase settings
   "TB.SETS TB GPIB.READ
   " CPL GND" AMP GPIB.WRITE  \ Ground the amplifier input
   " MOD PPA;SRC LIN" TB GPIB.WRITE  \ Auto trig on line signal

   AUTO.INTENSITY             \ Set intensity for a good trace
   ACQ.ATC                    \ Get a ground trace waveform
   MEAN                       \ Find average
   GND.LEVEL :=               \ Store ground level

   "AMP.SETS AMP GPIB.WRITE   \ Restore amplifier settings
   "TB.SETS TB GPIB.WRITE     \ Restore timebase settings
;

40 STRING "TEXT.BUF

\ Word to read and decode a numeric parameter from the digitizer.  Enter
\ with command header on string stack, return with numeric result on number
\ stack.  Can be used to read trace intensity and scale factors.
\
: GET.NUMERIC.PARM
   "DUP                       \ Save a copy of the command header
   " ?" "CAT DIG GPIB.WRITE   \ Append a '?' and send to digitizer
   "TEXT.BUF DIG GPIB.READ    \ Read string result back from dig.
   "TEXT.BUF "SKIP            \ Skip over header part of answer
   ASCII ; "NUMBER            \ Convert numeric string to number
;

REAL SCALAR VOLTS.PER.LEVEL   \ Volts per digitizer level
REAL SCALAR TIME.PER.POINT    \ Sample interval

\ Word to read vertical and horizontal scale factors from
\ digitizer mainframe.  This will work with MANUAL plugins too.
\
: GET.SCALE.FACT
  " VS1" GET.NUMERIC.PARM     \	Read Volts per division
  64 / VOLTS.PER.LEVEL :=     \ 64 levels per div
  " HS1" GET.NUMERIC.PARM     \ Read Time per division
  51.2 / TIME.PER.POINT :=    \ 51.2 samples per div
;


\ Arrays to store waveform data, fitted curve and residuals from fit.
\
REAL SCALAR N                 \ Fundamental record length
512 N :=
REAL DIM[ N ] ARRAY XX        \ Horizontal ramp array
REAL DIM[ N ] ARRAY WFM       \ Digitized waveform
REAL DIM[ N ] ARRAY FITTED    \ Fitted function
REAL DIM[ N ] ARRAY RESIDUALS \ Difference between acquired & fit
REAL DIM[ 2 ] ARRAY COEF      \ Coefficients of fitting function
INTEGER SCALAR START.INDEX    \ Beginning of decay part of data


\ Acquired signal has exponential growth and decay portions.  This word
\ determines where the decay portion begins in order to isolate it for
\ curve fitting.  Method is to examine signal slope determined from
\ smoothed first derivative.  Expects acquired waveform on stack and
\ returns with index of decay curve start on stack.
\ 
: FIND.DECAY.START
   1 SET.ORDER                \ Request first derivative
   2 SET.DEGREE               \ Using 2nd order polynomial
   WFM DIFFERENTIATE.DATA     \ Compute derivative
   0.2 SET.CUTOFF.FREQ        \ Low pass filter
   SMOOTH                     \ to smooth the result
   2 SET.#.OPTIMA             \ Request 2nd minima
   LOCAL.MINIMA               \ Search for it
   DROP                       \ Ignore value of minima
   [ 2 ]                      \ Return index of 2nd minima
;

\ This word performs the exponential curve fit and evaluates the
\ fitted function at the same X values as acquired data.  Requires
\ START.INDEX and N as limit indices for curve fit.
\
: COMPUTE.CURVE
   XX []RAMP                  \ Create linear ramp (1 .. N) 
   XX SUB[ START.INDEX , N ]  \ Select decay portion of ramp
   WFM SUB[ START.INDEX , N ] \ Select decay portion of data
   LEASTSQ.EXP.FIT            \ Perform the curve fit 
   COEF :=                    \ Store  coefficients of fitting function
   FITTED []RAMP              \ Compute fitted curve
   FITTED COEF [ 1 ] * COEF [ 2 ] +
   EXP FITTED :=              \ Store it
;


\ This word computes the difference between the acquired data and
\ the fitted curve.
\
: COMPUTE.RESIDUALS
   FITTED WFM -               \ Subtract observed from calculated
   RESIDUALS :=               \ Store in array RESIDUALS
;


\ Words to define the graphics.
\
16 VIDEO.MODE
16 GRAPHICS.DISPLAY.MODE
\
0.8 0.7 AXIS.SIZE   \ Leave room at top of graphs for title
1 1 TICK.JUST       \ Keep tick marks outside of graph  
\
\ Word to print a centered title at top of current graph.
\
: "PLOT.TITLE
     NORMAL.COORDS            \ Use normalized coordinates (0-1)
     0.5 0.91 POSITION        \ Center the cursor at top of graph
     CENTERED.LABEL           \ Display the title
;

\ Words to plot results of our calculations.  Upper part of display
\ shows acquired & fit data with the coefficients of fitting function.
\ Lower half displays residuals from curve fit.
\
: DISPLAY.DECAY
   0.2 0.3 VUPORT.ORIG
   0.8 0.7 VUPORT.SIZE
   1 VUPORT.COLOR             \ Blue
   11 AXIS.COLOR              \ Light Cyan
   11 LABEL.COLOR             \ Light Cyan
   14 COLOR                   \ Yellow
   WFM Y.AUTO.PLOT            \ Plot acquired data
   15 COLOR                   \ Intensified White
   FITTED Y.DATA.PLOT         \ Plot fitted curve
   OUTLINE                    \ Frame the plot
   "      Acquired Data & Fitted Curve" "PLOT.TITLE   \ Add title
;
\
: DISPLAY.RESIDUALS           \ Plot Residuals versus Fitted Curve
   0.2 0.0 VUPORT.ORIG
   0.8 0.3 VUPORT.SIZE
   5 VUPORT.COLOR             \ Magenta (Purple)
   15 COLOR                   \ Intensified White
   DOTTED                     \ Use dots at each point instead of line
   HORIZONTAL GRID.OFF        \ Turn off grid
   VERTICAL GRID.OFF
   FITTED                     \ Fitted Curve
   SUB[ START.INDEX , N ]     \ Select decay curve portion
   RESIDUALS                  \ Residuals
   SUB[ START.INDEX , N ]     \ Select decay curve portion
   XY.AUTO.PLOT               \ Display the data
   OUTLINE                    \ Frame the plot
   " Residuals versus Fitted Curve" "PLOT.TITLE  \ Add title
;
\
: DISPLAY.COEFS               \ Display fitting function & coefficients
   0.72 0.72 VUPORT.ORIG
   0.20 0.18 VUPORT.SIZE
   0 VUPORT.COLOR
   VUPORT.CLEAR               \ Erase current viewport
   15 COLOR
   OUTLINE                    \ Frame it
   NORMAL.COORDS              \ Work in (0-1) coordinates
   0.17 0.75 POSITION         \ Top line of text
   " Y = e" LABEL
   0.55 0.85 POSITION
   " ax + b" LABEL
   0.17 0.5 POSITION          \ Middle line
   " a =" COEF [ 1 ] "." "CAT LABEL
   0.17 0.25 POSITION         \ Bottom line
   " b =" COEF [ 2 ] "." "CAT LABEL
;


\ This word performs the entire application.  Ground reference is captured,
\ then decay signal is captured and analyzed.  Results displayed graphically.
\
: ANALYZE.EXP.DECAY 
   " GRAT OFF" DIG GPIB.WRITE \ Make sure the graticule is off
   GET.GND.LEVEL              \ Acquire ground reference
   AUTO.INTENSITY             \ Adjust the trace intensity
   ACQ.ATC                    \ Acquire the decay signal
   GND.LEVEL -                \ Reference to ground
   2 /                        \ Correct for factor of 2 from ATC
   WFM :=                     \ Save it
   " MODE TV" DIG GPIB.WRITE  \ Switch 7912 to real time video display
   FIND.DECAY.START           \ Locate beginning of data of interest
   START.INDEX :=             \ Save it
   COMPUTE.CURVE              \ Perform the curve fit
   COMPUTE.RESIDUALS          \ Determine the difference
   GRAPHICS.DISPLAY           \ Switch to graphics display
   DISPLAY.DECAY              \ Display acquired data & fit curve
   DISPLAY.RESIDUALS          \ Display residuals curve
   DISPLAY.COEFS              \ Display coefficients of fit function
   CURSOR.OFF                 \ Blank the graphics cursor
;


\ These function key assignments can be handy.
\
\  F1  - Perform Application
\  F9  - Erase command area of screen
\  F10 - Copy graphics screen on printer
\
59 FUNCTION.KEY.DOES ANALYZE.EXP.DECAY
67 FUNCTION.KEY.DOES SCREEN.CLEAR
68 FUNCTION.KEY.DOES SCREEN.PRINT
